/**
 * Backup & Migration Pro - Admin JavaScript
 */

(function($) {
    'use strict';

    const JBM = {
        
        initialized: false,
        
        /**
         * Initialisierung
         */
        init: function() {
            // Verhindere mehrfache Initialisierung
            if (this.initialized) {
                console.log('BMP bereits initialisiert, überspringe...');
                return;
            }
            
            console.log('BMP wird initialisiert...');
            this.initialized = true;
            
            this.bindEvents();
            this.initUploadArea();
            this.updateCompressionValue();
            this.initBackupTypeCards();
            
            console.log('BMP erfolgreich initialisiert!');
        },
        
        /**
         * Event-Handler binden
         */
        bindEvents: function() {
            console.log('Binde Event-Handler...');
            
            // Backup erstellen
            $(document).off('submit', '#jbm-create-backup-form').on('submit', '#jbm-create-backup-form', this.createBackup.bind(this));
            console.log('Form-Submit-Handler gebunden');
            
            // Test: Prüfe ob Formular existiert
            const form = $('#jbm-create-backup-form');
            console.log('Formular gefunden:', form.length);
            
            // Backup löschen
            $(document).off('click', '.jbm-delete-backup').on('click', '.jbm-delete-backup', this.deleteBackup.bind(this));
            
            // Backup-Schutz umschalten
            $(document).off('click', '.jbm-toggle-protection').on('click', '.jbm-toggle-protection', this.toggleProtection.bind(this));
            
            // Backup wiederherstellen
            $(document).off('click', '.jbm-restore-backup').on('click', '.jbm-restore-backup', this.restoreBackup.bind(this));
            
            // Backup herunterladen (mit Fehlerbehandlung)
            $(document).off('click', '.jbm-download-backup').on('click', '.jbm-download-backup', this.handleDownload.bind(this));
            
            // Backups aktualisieren
            $(document).off('click', '#jbm-refresh-backups').on('click', '#jbm-refresh-backups', this.refreshBackups.bind(this));
            
            // Zeitplan speichern
            $(document).off('submit', '#jbm-schedule-form').on('submit', '#jbm-schedule-form', this.saveSchedule.bind(this));
            
            // Zeitplan aktivieren/deaktivieren
            $(document).off('change', '#schedule-enabled').on('change', '#schedule-enabled', function() {
                $('#schedule-options').toggle(this.checked);
            });
            
            // Migration starten
            $(document).off('submit', '#jbm-migration-form').on('submit', '#jbm-migration-form', this.startMigration.bind(this));
            
            // Migration URL Vorschau
            $(document).off('input', '#new-url').on('input', '#new-url', function() {
                $('#preview-new-url').text($(this).val() || 'https://neue-domain.de');
            });
            
            $(document).off('input', '#old-url').on('input', '#old-url', function() {
                $('#preview-old-url').text($(this).val());
            });
            
            // Backup-Typ auswählen - nutze change-Event auf Radio-Buttons (funktioniert mit Labels)
            $(document).off('change', 'input[name="backup_type"]').on('change', 'input[name="backup_type"]', function() {
                console.log('Radio-Button geändert:', $(this).val());
                $('.bmp-backup-type-card, .jbm-backup-type-card').removeClass('active');
                const card = $(this).closest('.bmp-backup-type-card, .jbm-backup-type-card');
                if (card.length) {
                    card.addClass('active');
                    console.log('Backup-Typ geändert zu:', $(this).val());
                } else {
                    console.warn('Card nicht gefunden für Radio-Button');
                }
            });
            console.log('Radio-Button change-Handler gebunden');
            
            // Klick auf Card (für visuelles Feedback)
            $(document).off('click', '.bmp-backup-type-card, .jbm-backup-type-card').on('click', '.bmp-backup-type-card, .jbm-backup-type-card', function(e) {
                console.log('Card geklickt');
                // Wenn direkt auf Radio-Button geklickt, nichts tun (Label macht das)
                if ($(e.target).is('input[type="radio"]')) {
                    return true;
                }
                // Wenn auf Card geklickt, Radio-Button aktivieren
                const radio = $(this).find('input[type="radio"]');
                if (radio.length) {
                    if (!radio.is(':checked')) {
                        radio.prop('checked', true).trigger('change');
                        console.log('Radio-Button aktiviert:', radio.val());
                    }
                } else {
                    console.warn('Radio-Button nicht gefunden in Card');
                }
            });
            console.log('Card click-Handler gebunden');
            
            // Test: Prüfe ob Cards existieren
            const cards = $('.bmp-backup-type-card, .jbm-backup-type-card');
            console.log('Backup-Type-Cards gefunden:', cards.length);
            
            // E-Mail Benachrichtigungen
            $(document).off('change', '#email-notifications').on('change', '#email-notifications', function() {
                $('#email-address-group').toggle(this.checked);
            });
            
            // Kompressionswert anzeigen
            $(document).off('input', '#compression-level').on('input', '#compression-level', this.updateCompressionValue);
            
            // Logs löschen
            $(document).off('click', '#jbm-clear-logs').on('click', '#jbm-clear-logs', this.clearLogs.bind(this));
        },
        
        /**
         * Backup-Type-Cards initialisieren
         */
        initBackupTypeCards: function() {
            // Setze die active-Klasse auf die Card mit dem gecheckten Radio-Button
            $('.bmp-backup-type-card, .jbm-backup-type-card').each(function() {
                const $card = $(this);
                const $radio = $card.find('input[type="radio"]');
                if ($radio.is(':checked')) {
                    $card.addClass('active');
                } else {
                    $card.removeClass('active');
                }
            });
        },
        
        /**
         * Upload-Bereich initialisieren
         */
        initUploadArea: function() {
            const uploadArea = $('#jbm-upload-area');
            
            if (!uploadArea.length) {
                console.log('Upload-Bereich nicht gefunden auf dieser Seite');
                return;
            }
            
            // Alle Events vorher entfernen (verhindert Rekursion)
            uploadArea.off();
            $('#backup-file').off();
            $(document).off('click', '#jbm-upload-start');
            
            // Datei-Input direkt ansprechen wenn auf Upload-Area geklickt wird
            uploadArea.on('click', function(e) {
                const target = e.target;
                const tagName = target.tagName.toLowerCase();
                
                // Nicht triggern wenn es ein Button oder Link ist
                if (tagName !== 'button' && tagName !== 'a' && tagName !== 'input') {
                    document.getElementById('backup-file').click();
                }
            });
            
            // Drag & Drop für mehrere Dateien
            uploadArea.on('dragover', function(e) {
                e.preventDefault();
                e.stopPropagation();
                uploadArea.addClass('dragover');
            });
            
            uploadArea.on('dragleave', function(e) {
                e.preventDefault();
                e.stopPropagation();
                uploadArea.removeClass('dragover');
            });
            
            uploadArea.on('drop', function(e) {
                e.preventDefault();
                e.stopPropagation();
                uploadArea.removeClass('dragover');
                
                const files = e.originalEvent.dataTransfer.files;
                if (files.length > 0) {
                    document.getElementById('backup-file').files = files;
                    JBM.showSelectedFiles(files);
                }
            });
            
            // Datei-Auswahl Event
            $('#backup-file').on('change', function(e) {
                const files = e.target.files;
                if (files && files.length > 0) {
                    JBM.showSelectedFiles(files);
                }
            });
            
            // Upload-Start Button
            $(document).on('click', '#jbm-upload-start', function(e) {
                e.preventDefault();
                JBM.uploadMultipleBackups();
            });
            
            console.log('Upload-Bereich erfolgreich initialisiert');
        },
        
        /**
         * Zeigt ausgewählte Dateien an
         */
        showSelectedFiles: function(files) {
            const container = $('#jbm-selected-files');
            const fileList = $('#jbm-file-list');
            
            let html = '<div class="bmp-file-items">';
            let totalSize = 0;
            
            for (let i = 0; i < files.length; i++) {
                const file = files[i];
                const size = JBM.formatBytes(file.size);
                totalSize += file.size;
                
                html += '<div class="bmp-file-item">';
                html += '<span class="dashicons dashicons-media-archive"></span>';
                html += '<span class="bmp-file-name">' + JBM.escapeHtml(file.name) + '</span>';
                html += '<span class="bmp-file-size">' + size + '</span>';
                html += '</div>';
            }
            
            html += '</div>';
            html += '<div style="margin-top: 10px; padding-top: 10px; border-top: 1px solid #ddd;">';
            html += '<strong>' + jbmAdmin.strings.total + '</strong> ' + files.length + ' ' + jbmAdmin.strings.files_count + ', ' + JBM.formatBytes(totalSize);
            html += '</div>';
            
            fileList.html(html);
            container.fadeIn(300);
        },
        
        /**
         * Lädt mehrere Backups in Batches hoch (für viele Dateien)
         */
        uploadMultipleBackups: function() {
            const files = $('#backup-file')[0].files;
            if (!files || files.length === 0) {
                JBM.showNotification('Keine Dateien ausgewählt', 'error');
                return;
            }
            
            const button = $('#jbm-upload-start');
            button.prop('disabled', true);
            
            const totalFiles = files.length;
            console.log('🚀 Starte SEQUENZIELLEN Upload für ' + totalFiles + ' Datei(en)...');
            console.log('📝 Eine Datei nach der anderen - umgeht Server-Limits!');
            
            // IMMER sequenziell - egal wie viele Dateien
            JBM.uploadSequentially(files, button);
        },
        
        /**
         * Lädt Dateien SEQUENZIELL hoch (eine nach der anderen)
         * Umgeht: 413 Entity Too Large, max_file_uploads, Timeouts
         */
        uploadSequentially: function(files, button) {
            const totalFiles = files.length;
            let currentIndex = 0;
            const uploadedFiles = [];
            
            // Progress-Box einblenden
            $('#jbm-selected-files').fadeOut(300, function() {
                const progressHtml = '<div id="bmp-sequential-progress" style="padding: 20px; background: #f6f7f7; border-radius: 6px;">' +
                    '<h4 style="margin: 0 0 15px 0;">📤 ' + jbmAdmin.strings.upload_running + '</h4>' +
                    '<div class="bmp-progress-bar" style="height: 12px; background: #e5e5e5; border-radius: 6px; overflow: hidden; margin-bottom: 10px;">' +
                    '<div id="bmp-seq-progress-bar" style="height: 100%; background: linear-gradient(90deg, #2271b1, #72aee6); width: 0%; transition: width 0.3s ease;"></div>' +
                    '</div>' +
                    '<p id="bmp-seq-status" style="margin: 0; text-align: center; font-weight: 600; color: #1d2327;">' + jbmAdmin.strings.file_x_of_y.replace('%d', '0').replace('%d', totalFiles) + '</p>' +
                    '<p style="margin: 10px 0 0 0; text-align: center; font-size: 12px; color: #646970;">' + jbmAdmin.strings.please_dont_close + '</p>' +
                    '</div>';
                
                $('#jbm-upload-backup-form').append(progressHtml);
            });
            
            const uploadNext = function() {
                if (currentIndex >= totalFiles) {
                    // Fertig!
                    console.log('✅ Alle Dateien hochgeladen:', uploadedFiles);
                    $('#bmp-seq-status').html('<span style="color: #00a32a;">✅ ' + jbmAdmin.strings.all_files_uploaded.replace('%d', totalFiles) + '</span>');
                    
                    JBM.showNotification('Alle Dateien erfolgreich hochgeladen!', 'success');
                    
                    setTimeout(function() {
                        location.reload();
                    }, 2000);
                    return;
                }
                
                const file = files[currentIndex];
                const fileNum = currentIndex + 1;
                
                console.log('📤 Lade Datei ' + fileNum + '/' + totalFiles + ': ' + file.name);
                
                // Progress aktualisieren
                const percent = Math.round((currentIndex / totalFiles) * 100);
                $('#bmp-seq-progress-bar').css('width', percent + '%');
                $('#bmp-seq-status').text(jbmAdmin.strings.file_uploading.replace('%d', fileNum).replace('%d', totalFiles).replace('%s', file.name));
                
                // Upload mit Chunking (für große Dateien)
                JBM.uploadFileInChunks(file, fileNum, totalFiles, function(success, filePath) {
                    if (success) {
                        uploadedFiles.push(filePath);
                        currentIndex++;
                        
                        // Nächste Datei nach kurzer Pause
                        setTimeout(uploadNext, 500);
                    } else {
                        JBM.showNotification('Upload-Fehler bei Datei ' + fileNum, 'error');
                        button.prop('disabled', false);
                        $('#jbm-sequential-progress').remove();
                        $('#jbm-selected-files').fadeIn(300);
                    }
                });
            };
            
            // Ersten Upload starten
            uploadNext();
        },
        
        /**
         * Lädt eine Datei in Chunks hoch (umgeht Nginx client_max_body_size)
         * @param {File} file - Die hochzuladende Datei
         * @param {int} fileNum - Nummer der Datei
         * @param {int} totalFiles - Gesamtanzahl Dateien
         * @param {function} callback - Callback(success, filePath)
         */
        uploadFileInChunks: function(file, fileNum, totalFiles, callback) {
            const chunkSize = 1024 * 1024; // 1 MB Chunks (umgeht jedes Server-Limit!)
            const totalChunks = Math.ceil(file.size / chunkSize);
            let currentChunk = 0;
            
            console.log('📦 Chunked Upload: ' + file.name + ' (' + JBM.formatBytes(file.size) + ') in ' + totalChunks + ' Chunks à 1MB');
            
            const uploadNextChunk = function() {
                if (currentChunk >= totalChunks) {
                    // Alle Chunks hochgeladen - fertig!
                    console.log('✅ Alle Chunks hochgeladen für: ' + file.name);
                    callback(true, file.name);
                    return;
                }
                
                const start = currentChunk * chunkSize;
                const end = Math.min(start + chunkSize, file.size);
                const chunk = file.slice(start, end);
                
                const formData = new FormData();
                formData.append('action', 'jbm_upload_chunk');
                formData.append('nonce', jbmAdmin.nonce);
                formData.append('file_chunk', chunk);
                formData.append('file_name', file.name);
                formData.append('chunk_index', currentChunk);
                formData.append('total_chunks', totalChunks);
                formData.append('file_size', file.size);
                
                console.log('  📤 Chunk ' + (currentChunk + 1) + '/' + totalChunks + ' (' + JBM.formatBytes(chunk.size) + ')');
                
                // Mini-Progress innerhalb der Datei
                const filePercent = Math.round((currentChunk / totalChunks) * 100);
                $('#bmp-seq-status').html(jbmAdmin.strings.file_uploading_progress.replace('%d', fileNum).replace('%d', totalFiles).replace('%s', file.name).replace('%s', filePercent));
                
                $.ajax({
                    url: jbmAdmin.ajaxUrl,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    timeout: 60000, // 1 Minute pro Chunk
                    success: function(response) {
                        if (response.success) {
                            currentChunk++;
                            uploadNextChunk(); // Nächster Chunk
                        } else {
                            console.error('❌ Chunk-Fehler:', response);
                            callback(false, null);
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('❌ Chunk-Upload-Fehler:', status, error);
                        callback(false, null);
                    }
                });
            };
            
            uploadNextChunk();
        },
        
        /**
         * Lädt Dateien in Batches hoch (für >10 Dateien)
         */
        uploadInBatches: function(files, button) {
            const totalFiles = files.length;
            const batchSize = 8; // Max 8 Dateien pro Batch (PHP max_file_uploads ist oft 20)
            const totalBatches = Math.ceil(totalFiles / batchSize);
            
            console.log('Batch-Upload: ' + totalFiles + ' Dateien in ' + totalBatches + ' Batches (à ' + batchSize + ' Dateien)');
            
            // Fortschrittsanzeige einblenden
            $('#jbm-selected-files').fadeOut(300, function() {
                $('#jbm-batch-progress').fadeIn(300);
            });
            
            $('#jbm-batch-total').text(totalBatches);
            $('#jbm-batch-current').text('0');
            
            JBM.showNotification('Lade ' + totalFiles + ' Dateien in ' + totalBatches + ' Batches hoch...', 'info');
            
            let currentBatch = 0;
            const uploadedPaths = [];
            
            const uploadNextBatch = function() {
                if (currentBatch >= totalBatches) {
                    // Alle Batches hochgeladen - jetzt zusammenführen
                    console.log('Alle Batches hochgeladen, führe zusammen...');
                    
                    $('#jbm-batch-progress-bar').css('width', '100%');
                    $('#jbm-batch-progress-text').text(jbmAdmin.strings.merging_files.replace('%d', uploadedPaths.length));
                    
                    JBM.mergeUploadedFiles(uploadedPaths, totalFiles, button);
                    return;
                }
                
                const startIdx = currentBatch * batchSize;
                const endIdx = Math.min(startIdx + batchSize, totalFiles);
                const batchFiles = Array.from(files).slice(startIdx, endIdx);
                
                console.log('Lade Batch ' + (currentBatch + 1) + '/' + totalBatches + ' (' + batchFiles.length + ' Dateien)');
                
                // Fortschritt aktualisieren
                const progress = Math.round((currentBatch / totalBatches) * 100);
                $('#jbm-batch-progress-bar').css('width', progress + '%');
                $('#jbm-batch-current').text(currentBatch + 1);
                $('#jbm-batch-progress-text').text(jbmAdmin.strings.file_uploading.replace('%d', currentBatch + 1).replace('%d', totalBatches).replace('%s', batchFiles.length + ' ' + jbmAdmin.strings.files || 'files'));
                
                JBM.uploadBatch(batchFiles, startIdx, endIdx, button, currentBatch + 1, totalBatches, function(paths) {
                    uploadedPaths.push(...paths);
                    currentBatch++;
                    
                    // Kurze Pause zwischen Batches (Server-Entlastung)
                    setTimeout(uploadNextBatch, 500);
                });
            };
            
            uploadNextBatch();
        },
        
        /**
         * Lädt ein einzelnes Batch hoch
         */
        uploadBatch: function(files, startIdx, endIdx, button, batchNum, totalBatches, callback) {
            const formData = new FormData();
            formData.append('action', 'jbm_upload_batch');
            formData.append('nonce', jbmAdmin.nonce);
            formData.append('batch_num', batchNum);
            formData.append('total_batches', totalBatches);
            
            // Dateien als Array (nicht FileList)
            const fileArray = Array.isArray(files) ? files : Array.from(files);
            
            for (let i = 0; i < fileArray.length; i++) {
                formData.append('backup_files[]', fileArray[i]);
            }
            
            $.ajax({
                url: jbmAdmin.ajaxUrl,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                timeout: 300000, // 5 Minuten Timeout
                xhr: function() {
                    const xhr = new window.XMLHttpRequest();
                    xhr.upload.addEventListener('progress', function(e) {
                        if (e.lengthComputable) {
                            const percent = Math.round((e.loaded / e.total) * 100);
                            console.log('Batch ' + batchNum + ' Upload: ' + percent + '%');
                        }
                    });
                    return xhr;
                },
                success: function(response) {
                    console.log('Batch ' + batchNum + ' Response:', response);
                    
                    if (response.success) {
                        const paths = response.data.uploaded_paths || [];
                        
                        if (callback) {
                            callback(paths);
                        } else {
                            // Letzter Batch oder einzelner Upload
                            JBM.showNotification(response.data.message, 'success');
                            JBM.refreshBackups();
                            $('#backup-file').val('');
                            $('#jbm-selected-files').fadeOut(300);
                            $('#jbm-batch-progress').fadeOut(300);
                            button.prop('disabled', false).html('<span class="dashicons dashicons-upload"></span> ' + jbmAdmin.strings.upload_start);
                        }
                    } else {
                        let errorMsg = 'FEHLER BEI BATCH ' + batchNum + ':\n\n';
                        errorMsg += (response.data.message || 'Unbekannter Fehler');
                        errorMsg += '\n\nMögliche Ursachen:\n';
                        errorMsg += '- Datei zu groß (Max: ' + JBM.getMaxUploadSize() + ')\n';
                        errorMsg += '- Ungültiges Dateiformat\n';
                        errorMsg += '- Schreibrechte fehlen\n';
                        errorMsg += '- PHP Memory Limit zu niedrig\n';
                        errorMsg += '\nLösung: Siehe Backup & Migration → Einstellungen → Logs';
                        
                        alert(errorMsg);
                        console.error('Batch Error Details:', response);
                        
                        JBM.showNotification(response.data.message || 'Fehler bei Batch ' + batchNum, 'error');
                        $('#jbm-batch-progress').fadeOut(300);
                        button.prop('disabled', false).html('<span class="dashicons dashicons-upload"></span> ' + jbmAdmin.strings.upload_start);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Batch ' + batchNum + ' AJAX Error:', xhr, status, error);
                    console.error('Response Text:', xhr.responseText);
                    
                    let errorMsg = 'KRITISCHER FEHLER BEI BATCH ' + batchNum + ':\n\n';
                    errorMsg += 'Status: ' + status + '\n';
                    errorMsg += 'Error: ' + error + '\n\n';
                    
                    if (xhr.responseText) {
                        const preview = xhr.responseText.substring(0, 500);
                        errorMsg += 'Server-Antwort (Auszug):\n' + preview + '\n\n';
                    }
                    
                    errorMsg += 'Mögliche Ursachen:\n';
                    errorMsg += '- PHP Upload-Limit erreicht\n';
                    errorMsg += '- Server-Timeout\n';
                    errorMsg += '- .htaccess blockiert Upload\n';
                    errorMsg += '- Ungültiger Dateityp\n';
                    errorMsg += '\nLösung: Browser-Konsole (F12) und Server-Logs prüfen';
                    
                    alert(errorMsg);
                    
                    JBM.showNotification('Kritischer Fehler bei Batch ' + batchNum, 'error');
                    $('#jbm-batch-progress').fadeOut(300);
                    button.prop('disabled', false).html('<span class="dashicons dashicons-upload"></span> ' + jbmAdmin.strings.upload_start);
                }
            });
        },
        
        /**
         * Führt hochgeladene Dateien zusammen
         */
        mergeUploadedFiles: function(uploadedPaths, totalFiles, button) {
            console.log('Führe ' + uploadedPaths.length + ' hochgeladene Dateien zusammen...');
            
            // Fortschritt auf 100% setzen (Upload fertig)
            $('#jbm-batch-progress-bar').css('width', '100%');
            $('#jbm-batch-progress-text').text(jbmAdmin.strings.merging_files.replace('%d', uploadedPaths.length));
            $('#jbm-batch-details').html('<em>' + jbmAdmin.strings.merging_may_take_time + '</em>');
            
            JBM.showNotification('Führe ' + uploadedPaths.length + ' Dateien zusammen...', 'info');
            
            $.ajax({
                url: jbmAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'jbm_merge_uploaded_files',
                    nonce: jbmAdmin.nonce,
                    file_paths: uploadedPaths
                },
                timeout: 600000, // 10 Minuten für Merge
                success: function(response) {
                    console.log('Merge Response:', response);
                    
                    if (response.success) {
                        $('#jbm-batch-progress-text').text('✅ ' + jbmAdmin.strings.merge_success);
                        
                        JBM.showNotification(
                            '✅ ' + totalFiles + ' Dateien erfolgreich hochgeladen und zusammengeführt!\n\n' +
                            '✨ Ein kombiniertes Backup wurde erstellt (' + JBM.formatBytes(response.data.merged_size) + ').',
                            'success'
                        );
                        
                        setTimeout(function() {
                            JBM.refreshBackups();
                            $('#backup-file').val('');
                            $('#jbm-batch-progress').fadeOut(300);
                            button.prop('disabled', false).html('<span class="dashicons dashicons-upload"></span> ' + jbmAdmin.strings.upload_start);
                        }, 2000);
                    } else {
                        $('#jbm-batch-progress-text').text('❌ ' + jbmAdmin.strings.merge_error);
                        
                        JBM.showNotification('Fehler beim Zusammenführen: ' + response.data.message, 'error');
                        
                        setTimeout(function() {
                            $('#jbm-batch-progress').fadeOut(300);
                            button.prop('disabled', false).html('<span class="dashicons dashicons-upload"></span> ' + jbmAdmin.strings.upload_start);
                        }, 3000);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Merge Error:', xhr, status, error);
                    
                    $('#jbm-batch-progress-text').text('❌ ' + jbmAdmin.strings.critical_error);
                    
                    JBM.showNotification('Fehler beim Zusammenführen: ' + error, 'error');
                    
                    setTimeout(function() {
                        $('#jbm-batch-progress').fadeOut(300);
                        button.prop('disabled', false).html('<span class="dashicons dashicons-upload"></span> ' + jbmAdmin.strings.upload_start);
                    }, 3000);
                }
            });
        },
        
        /**
         * Backup erstellen
         */
        // Speichert den Zeitpunkt des letzten Backup-Versuchs (clientseitig)
        lastBackupAttempt: 0,
        
        createBackup: function(e) {
            if (e) {
                e.preventDefault();
                e.stopPropagation();
            }
            
            // Formular direkt finden
            const form = $('#jbm-create-backup-form');
            
            if (!form.length) {
                console.error('Formular #jbm-create-backup-form nicht gefunden!');
                alert('Fehler: Formular nicht gefunden. Bitte Seite neu laden.');
                return false;
            }
            
            const button = form.find('button[type="submit"]');
            const progressContainer = $('#jbm-backup-progress');
            const progressBar = progressContainer.find('.jbm-progress-bar-fill');
            
            console.log('Backup-Erstellung gestartet...');
            console.log('Form gefunden:', form.length);
            console.log('Progress Container gefunden:', progressContainer.length);
            
            // Prüfe ob Backup-Typ ausgewählt ist
            const backupType = $('input[name="backup_type"]:checked').val();
            if (!backupType) {
                console.error('Kein Backup-Typ ausgewählt!');
                alert('Bitte wählen Sie einen Backup-Typ aus.');
                return false;
            }
            console.log('Backup-Typ:', backupType);
            
            // Clientseitige Rate-Limit-Prüfung (verhindert UI-Flackern)
            const now = Math.floor(Date.now() / 1000);
            const timeSinceLastAttempt = now - JBM.lastBackupAttempt;
            if (JBM.lastBackupAttempt > 0 && timeSinceLastAttempt < 30) {
                const waitTime = 30 - timeSinceLastAttempt;
                JBM.showRateLimitModal({
                    message: (jbmAdmin.strings.rate_limit_message || 'Bitte warten Sie ' + waitTime + ' Sekunden vor dem nächsten Backup').replace('%d', waitTime),
                    reason: jbmAdmin.strings.rate_limit_reason || 'Aus Sicherheitsgründen ist nur ein Backup alle 30 Sekunden erlaubt. Dies schützt Ihren Server vor Überlastung.',
                    wait_time: waitTime
                });
                return false;
            }
            
            // Button deaktivieren
            button.prop('disabled', true);
            form.fadeOut(300, function() {
                progressContainer.fadeIn(300);
            });
            
            // Motivierende Nachrichten einblenden
            $('#jbm-backup-messages').fadeIn(300);
            
            // Progress-Animation mit Prozentanzeige
            let progress = 0;
            const progressInterval = setInterval(function() {
                progress += Math.random() * 10;
                if (progress > 90) progress = 90;
                progressBar.css('width', progress + '%');
                $('.jbm-progress-percentage').text(Math.round(progress) + '%');
            }, 500);
            
            // Motivierende Nachrichten (alle 15 Sekunden wechseln)
            let messageIndex = 0;
            JBM.showBackupMessage(messageIndex);
            
            const messageInterval = setInterval(function() {
                messageIndex++;
                JBM.showBackupMessage(messageIndex);
            }, 15000); // Alle 15 Sekunden
            
            // Stoppe Intervals wenn Backup fertig
            const stopIntervals = function() {
                clearInterval(progressInterval);
                clearInterval(messageInterval);
            };
            
            const backupData = {
                action: 'jbm_create_backup',
                nonce: jbmAdmin.nonce,
                backup_name: $('#backup-name').val(),
                backup_type: $('input[name="backup_type"]:checked').val()
            };
            
            console.log('Backup-Daten:', backupData);
            
            // Zeitpunkt speichern (für clientseitige Rate-Limit-Prüfung)
            JBM.lastBackupAttempt = Math.floor(Date.now() / 1000);
            
            // AJAX-Request
            $.ajax({
                url: jbmAdmin.ajaxUrl,
                type: 'POST',
                data: backupData,
                success: function(response) {
                    console.log('Backup-Response:', response);
                    stopIntervals();
                    progressBar.css('width', '100%');
                    $('.jbm-progress-percentage').text('100%');
                    
                    // Erfolgsnachricht zeigen
                    JBM.showBackupMessage('success');
                    
                    // Spezielle Behandlung für Rate-Limiting - sofort ohne Verzögerung
                    if (!response.success && response.data && response.data.rate_limited) {
                        // Bei Rate-Limiting: Fortschrittsanzeige sofort verstecken, Modal zeigen
                        progressContainer.hide();
                        $('#jbm-backup-messages').hide();
                        form.show();
                        button.prop('disabled', false);
                        progressBar.css('width', '0%');
                        $('.jbm-progress-percentage').text('0%');
                        JBM.showRateLimitModal(response.data);
                        return;
                    }
                    
                    setTimeout(function() {
                        if (response.success) {
                            JBM.showNotification(jbmAdmin.strings.success, 'success');
                            JBM.refreshBackups();
                            JBM.updateAdminBarStatus(); // Live-Update der Admin-Bar
                            form[0].reset();
                            $('.bmp-backup-type-card, .jbm-backup-type-card').removeClass('active');
                            $('.bmp-backup-type-card:first, .jbm-backup-type-card:first').addClass('active');
                        } else {
                            JBM.showBackupMessage('error');
                            JBM.showNotification(response.data.message || jbmAdmin.strings.error, 'error');
                        }
                        
                        progressContainer.fadeOut(300, function() {
                            form.fadeIn(300);
                            button.prop('disabled', false);
                            progressBar.css('width', '0%');
                            $('.jbm-progress-percentage').text('0%');
                            $('#jbm-backup-messages').hide();
                        });
                    }, 2000);
                },
                error: function(xhr, status, error) {
                    console.error('AJAX Error:', xhr, status, error);
                    console.error('Response Text:', xhr.responseText);
                    
                    stopIntervals();
                    JBM.showBackupMessage('error');
                    
                    let errorMessage = jbmAdmin.strings.error;
                    if (xhr.responseText) {
                        try {
                            const jsonResponse = JSON.parse(xhr.responseText);
                            if (jsonResponse.data && jsonResponse.data.message) {
                                errorMessage = jsonResponse.data.message;
                            }
                        } catch (e) {
                            // Nicht JSON, verwende Standard-Fehlermeldung
                            console.error('Response ist nicht JSON:', xhr.responseText.substring(0, 200));
                        }
                    }
                    
                    JBM.showNotification(errorMessage, 'error');
                    
                    setTimeout(function() {
                        progressContainer.fadeOut(300, function() {
                            form.fadeIn(300);
                            button.prop('disabled', false);
                            progressBar.css('width', '0%');
                            $('.jbm-progress-percentage').text('0%');
                            $('#jbm-backup-messages').hide();
                        });
                    }, 3000);
                }
            });
            
            return false; // Verhindere normales Form-Submit
        },
        
        /**
         * Backup hochladen (Legacy - nicht mehr verwendet, Multi-Upload übernimmt)
         */
        uploadBackup: function(e) {
            // Diese Funktion wird nicht mehr direkt aufgerufen
            // Alle Uploads gehen über uploadMultipleBackups
            // Hier als Fallback belassen
        },
        
        /**
         * Backup herunterladen (mit Fehlerbehandlung und Status-Überwachung)
         */
        handleDownload: function(e) {
            const link = $(e.currentTarget);
            const backupId = link.data('backup-id');
            const backupSize = link.data('backup-size') || 0;
            const backupSizeMB = backupSize / (1024 * 1024);
            
            // Warnung bei sehr großen Dateien (>500MB)
            if (backupSizeMB > 500) {
                if (!confirm(jbmAdmin.strings.large_file_warning.replace('%s', this.formatBytes(backupSize)))) {
                    e.preventDefault();
                    return false;
                }
            }
            
            // Download-Status anzeigen
            const originalText = link.html();
            link.html('<span class="dashicons dashicons-update spin"></span> ' + jbmAdmin.strings.starting_download);
            link.addClass('bmp-downloading');
            
            // Timeout für Download-Überwachung (30 Sekunden)
            const downloadTimeout = setTimeout(function() {
                // Prüfe ob Download noch läuft
                link.html(originalText);
                link.removeClass('bmp-downloading');
            }, 30000);
            
            // Versuche Download zu überwachen
            // Da es ein direkter Link ist, können wir nur den Start überwachen
            // Der Browser übernimmt dann den Rest
            
            // Fehlerbehandlung: Wenn der Link nach 5 Sekunden noch "lädt", könnte ein Fehler vorliegen
            setTimeout(function() {
                // Prüfe ob Seite noch lädt (könnte ein Fehler sein)
                // Aber wir können nicht direkt prüfen, ob der Download erfolgreich war
                // Der Browser zeigt Fehler normalerweise selbst an
            }, 5000);
            
            // Cleanup nach erfolgreichem Klick
            setTimeout(function() {
                clearTimeout(downloadTimeout);
                link.html(originalText);
                link.removeClass('bmp-downloading');
            }, 2000);
            
            // Erlaube normalen Link-Klick
            return true;
        },
        
        /**
         * Backup löschen
         */
        deleteBackup: function(e) {
            e.preventDefault();
            
            const button = $(e.currentTarget);
            const backupId = button.data('backup-id');
            // WICHTIG: Template verwendet 'bmp-backup-item', nicht 'jbm-backup-item'
            const backupItem = button.closest('.bmp-backup-item');
            
            if (backupItem.length === 0) {
                console.error('Backup-Item nicht gefunden!');
                return;
            }
            
            // Modal anzeigen
            const modal = $('#jbm-delete-modal');
            modal.find('#jbm-delete-modal-title').text(jbmAdmin.strings.delete_confirm_title);
            modal.find('#jbm-delete-modal-message').text(jbmAdmin.strings.delete_confirm_message);
            modal.find('#jbm-delete-modal-warning p').text(jbmAdmin.strings.delete_warning);
            modal.find('#jbm-delete-modal-confirm .confirm-text').text(jbmAdmin.strings.delete_confirm_button);
            modal.find('#jbm-delete-modal-cancel .cancel-text').text(jbmAdmin.strings.delete_cancel_button);
            
            // Modal anzeigen
            modal.css('display', 'flex');
            
            // Event-Handler für Bestätigung
            $('#jbm-delete-modal-confirm').off('click').on('click', function() {
                modal.css('display', 'none');
                
                // Jetzt wirklich löschen
                button.prop('disabled', true);
                
                $.ajax({
                    url: jbmAdmin.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'jbm_delete_backup',
                        nonce: jbmAdmin.nonce,
                        backup_id: backupId
                    },
                    success: function(response) {
                        if (response.success) {
                            // Backup-Element sofort aus der Liste entfernen
                            backupItem.fadeOut(300, function() {
                                $(this).remove();
                                
                                // Backup-Anzahl in der Überschrift aktualisieren
                                const remainingBackups = $('#jbm-backups-list .bmp-backup-item').length;
                                const header = $('.jbm-backups-count-header');
                                if (header.length > 0) {
                                    header.text(jbmAdmin.strings.existing_backups + ' (' + remainingBackups + ')');
                                }
                                
                                // Empty State anzeigen, wenn keine Backups mehr vorhanden
                                // WICHTIG: Template verwendet 'bmp-backup-item' und '#jbm-backups-list'
                                if (remainingBackups === 0) {
                                    $('#jbm-backups-list').html(
                                        '<div class="bmp-empty-state">' +
                                        '<span class="dashicons dashicons-cloud"></span>' +
                                        '<h3>' + jbmAdmin.strings.no_backups + '</h3>' +
                                        '<p>' + jbmAdmin.strings.create_first_backup + '</p>' +
                                        '</div>'
                                    );
                                }
                            });
                            JBM.showNotification(response.data.message, 'success');
                            JBM.updateAdminBarStatus(); // Live-Update der Admin-Bar
                        } else {
                            JBM.showNotification(response.data.message, 'error');
                            button.prop('disabled', false);
                        }
                    },
                    error: function() {
                        JBM.showNotification(jbmAdmin.strings.error_deleting, 'error');
                        button.prop('disabled', false);
                    }
                });
            });
            
            // Event-Handler für Abbrechen
            $('#jbm-delete-modal-cancel').off('click').on('click', function() {
                modal.css('display', 'none');
            });
            
            // Modal schließen bei Klick außerhalb
            modal.off('click').on('click', function(e) {
                if ($(e.target).is(modal)) {
                    modal.css('display', 'none');
                }
            });
        },
        
        /**
         * Backup-Schutz umschalten
         */
        toggleProtection: function(e) {
            e.preventDefault();
            
            const button = $(e.currentTarget);
            const backupId = button.data('backup-id');
            // WICHTIG: Template verwendet 'bmp-backup-item', nicht 'jbm-backup-item'
            const backupItem = button.closest('.bmp-backup-item');
            
            if (backupItem.length === 0) {
                console.error('Backup-Item nicht gefunden!');
                return;
            }
            
            button.prop('disabled', true);
            
            $.ajax({
                url: jbmAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'jbm_toggle_protection',
                    nonce: jbmAdmin.nonce,
                    backup_id: backupId
                },
                success: function(response) {
                    if (response.success) {
                        const isProtected = response.data.protected;
                        
                        // UI aktualisieren
                        backupItem.attr('data-protected', isProtected ? '1' : '0');
                        
                        if (isProtected) {
                            backupItem.addClass('bmp-backup-protected');
                            button.removeClass('bmp-button-secondary').addClass('bmp-button-warning');
                            button.find('.dashicons').removeClass('dashicons-lock').addClass('dashicons-unlock');
                            button.find('.protection-text').text(jbmAdmin.strings.unlock);
                            button.attr('title', jbmAdmin.strings.remove_protection);
                            
                            // Icon ändern - Template verwendet 'bmp-backup-icon'
                            backupItem.find('.bmp-backup-icon .dashicons')
                                .removeClass('dashicons-archive')
                                .addClass('dashicons-lock')
                                .css('color', '#d63638');
                            
                            // Badge hinzufügen - sofort anzeigen
                            // Entferne alte Badge falls vorhanden
                            backupItem.find('.jbm-protected-badge').remove();
                            
                            // Finde das h4-Element und füge Badge hinzu
                            const h4Element = backupItem.find('.bmp-backup-info h4');
                            if (h4Element.length > 0) {
                                h4Element.append(
                                    '<span class="jbm-protected-badge" style="display: inline-block; margin-left: 8px; padding: 2px 8px; background: #d63638; color: white; border-radius: 3px; font-size: 11px; font-weight: 600;">🔒 ' + jbmAdmin.strings.protected + '</span>'
                                );
                            } else {
                                console.error('h4-Element nicht gefunden für Badge!');
                            }
                            
                            // Löschen-Button deaktivieren
                            backupItem.find('.jbm-delete-backup').prop('disabled', true).attr('title', jbmAdmin.strings.protected_cannot_delete);
                        } else {
                            backupItem.removeClass('bmp-backup-protected');
                            button.removeClass('bmp-button-warning').addClass('bmp-button-secondary');
                            button.find('.dashicons').removeClass('dashicons-unlock').addClass('dashicons-lock');
                            button.find('.protection-text').text(jbmAdmin.strings.protect);
                            button.attr('title', jbmAdmin.strings.protect_from_deletion);
                            
                            // Icon zurücksetzen - Template verwendet 'bmp-backup-icon'
                            backupItem.find('.bmp-backup-icon .dashicons')
                                .removeClass('dashicons-lock')
                                .addClass('dashicons-archive')
                                .css('color', '');
                            
                            // Badge entfernen
                            backupItem.find('.jbm-protected-badge').remove();
                            
                            // Löschen-Button aktivieren
                            backupItem.find('.jbm-delete-backup').prop('disabled', false).removeAttr('title');
                        }
                        
                        JBM.showNotification(response.data.message, 'success');
                    } else {
                        JBM.showNotification(response.data.message || 'Fehler beim Aktualisieren', 'error');
                    }
                    button.prop('disabled', false);
                },
                error: function() {
                    JBM.showNotification('Fehler beim Aktualisieren', 'error');
                    button.prop('disabled', false);
                }
            });
        },
        
        /**
         * Backup wiederherstellen
         */
        restoreBackup: function(e) {
            e.preventDefault();
            
            const button = $(e.currentTarget);
            const backupId = button.data('backup-id');
            const backupItem = button.closest('.bmp-backup-item, .jbm-backup-item');
            
            // Bestätigungsdialog
            const confirmMessage = jbmAdmin.strings.confirm_restore || 
                'Möchten Sie dieses Backup wirklich wiederherstellen? Die aktuelle Website wird überschrieben.';
            
            // Restore-Bestätigungsdialog
            const wpconfigModal = $(
                '<div id="jbm-restore-wpconfig-modal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0, 0, 0, 0.7); z-index: 100000; align-items: center; justify-content: center;">' +
                    '<div style="background: white; border-radius: 12px; padding: 30px; max-width: 550px; width: 90%; box-shadow: 0 8px 32px rgba(0,0,0,0.3); position: relative;">' +
                        '<div style="text-align: center; margin-bottom: 20px;">' +
                            '<div style="font-size: 48px; margin-bottom: 40px;">⚠️</div>' +
                            '<h2 style="margin: 0 0 15px 0; color: #d63638; font-size: 22px;">' + (jbmAdmin.strings.confirm_restore_title || jbmAdmin.strings.confirm_restore || 'Backup wirklich wiederherstellen?') + '</h2>' +
                        '</div>' +
                        '<div style="margin-bottom: 20px;">' +
                            '<p style="margin: 0 0 15px 0; font-size: 15px; line-height: 1.6; color: #1d2327;">' + confirmMessage + '</p>' +
                            '<div style="background: #e7f3ff; border-left: 4px solid #2271b1; padding: 12px 15px; border-radius: 4px; margin-top: 15px;">' +
                                '<p style="margin: 0; font-size: 14px; color: #005177; line-height: 1.5;">' +
                                    '<strong>ℹ️ ' + (jbmAdmin.strings.hint || 'Hinweis:') + '</strong><br>' +
                                    (jbmAdmin.strings.wpconfig_not_restored || 'Die wp-config.php wird nicht wiederhergestellt. Die bestehende wp-config.php des Servers bleibt erhalten (beinhaltet korrekte Datenbank-Credentials).') +
                                '</p>' +
                            '</div>' +
                            '<div style="margin-top: 20px; padding: 15px; background: #f0f6fc; border: 1px solid #c3d4e6; border-radius: 4px;">' +
                                '<label style="display: flex; align-items: flex-start; cursor: pointer; gap: 10px;">' +
                                    '<input type="checkbox" id="jbm-keep-existing-plugins" style="margin-top: 3px; cursor: pointer;" />' +
                                    '<div style="flex: 1;">' +
                                        '<strong style="display: block; margin-bottom: 5px; font-size: 14px; color: #1d2327;">' + (jbmAdmin.strings.keep_existing_plugins_label || 'Bestehende Plugins behalten') + '</strong>' +
                                        '<span style="font-size: 13px; color: #50575e; line-height: 1.5;">' + (jbmAdmin.strings.keep_existing_plugins_desc || 'Plugins, die nicht im Backup enthalten sind, werden standardmäßig entfernt. Aktivieren Sie diese Option, um alle bestehenden Plugins zu behalten.') + '</span>' +
                                    '</div>' +
                                '</label>' +
                            '</div>' +
                        '</div>' +
                        '<div style="display: flex; gap: 10px; justify-content: flex-end;">' +
                            '<button id="jbm-restore-modal-cancel" class="bmp-button bmp-button-secondary" style="min-width: 120px;">' +
                                '<span class="dashicons dashicons-no"></span> ' +
                                '<span>' + (jbmAdmin.strings.delete_cancel_button || 'Abbrechen') + '</span>' +
                            '</button>' +
                            '<button id="jbm-restore-modal-confirm" class="bmp-button bmp-button-danger" style="min-width: 120px;">' +
                                '<span class="dashicons dashicons-update"></span> ' +
                                '<span>' + (jbmAdmin.strings.restore || 'Wiederherstellen') + '</span>' +
                            '</button>' +
                        '</div>' +
                    '</div>' +
                '</div>'
            );
            
            // Modal entfernen falls bereits vorhanden
            $('#jbm-restore-wpconfig-modal').remove();
            $('body').append(wpconfigModal);
            
            const modal = $('#jbm-restore-wpconfig-modal');
            modal.css('display', 'flex');
            
            // Cancel-Button
            $('#jbm-restore-modal-cancel').off('click').on('click', function() {
                modal.remove();
            });
            
            // Confirm-Button
            $('#jbm-restore-modal-confirm').off('click').on('click', function() {
                // wp-config.php Option wurde entfernt - immer false übergeben (wird ohnehin ignoriert)
                const overwriteWpconfig = false;
                // Checkbox-Wert lesen: false = entfernen (default), true = behalten
                const keepExistingPlugins = $('#jbm-keep-existing-plugins').is(':checked');
                modal.remove();
                
                // Restore durchführen
                JBM.executeRestore(backupId, backupItem, button, overwriteWpconfig, keepExistingPlugins);
            });
            
            // Modal schließen bei Klick außerhalb
            modal.off('click').on('click', function(e) {
                if ($(e.target).is(modal)) {
                    modal.remove();
                }
            });
        },
        
        executeRestore: function(backupId, backupItem, button, overwriteWpconfig, keepExistingPlugins) {
            // Erstelle Fortschrittsbalken-Container (falls nicht vorhanden)
            let progressContainer = $('#jbm-restore-progress');
            if (!progressContainer.length) {
                progressContainer = $(
                    '<div id="jbm-restore-progress" class="bmp-progress-container" style="display: none; margin-top: 20px;">' +
                        '<div class="bmp-progress-bar" style="position: relative;">' +
                            '<div class="jbm-restore-progress-bar-fill" style="width: 0%; height: 100%; background: linear-gradient(90deg, #2271b1 0%, #135e96 100%); border-radius: 4px; transition: width 0.3s ease;"></div>' +
                            '<div class="jbm-restore-progress-percentage" style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); font-weight: 700; color: white; text-shadow: 0 1px 2px rgba(0,0,0,0.5); z-index: 10;">0%</div>' +
                        '</div>' +
                        '<p class="bmp-progress-text" style="margin-top: 10px; text-align: center; color: #646970;">' + jbmAdmin.strings.restoring + '</p>' +
                    '</div>'
                );
                backupItem.after(progressContainer);
            }
            
            // Button deaktivieren und verstecken
            button.prop('disabled', true);
            backupItem.find('.bmp-button, .jbm-button').prop('disabled', true);
            
            // Fortschrittsbalken anzeigen
            progressContainer.fadeIn(300);
            const progressBar = progressContainer.find('.jbm-restore-progress-bar-fill');
            const progressPercentage = progressContainer.find('.jbm-restore-progress-percentage');
            
            // Progress-Animation mit Prozentanzeige
            let progress = 0;
            const progressInterval = setInterval(function() {
                progress += Math.random() * 8;
                if (progress > 90) progress = 90;
                progressBar.css('width', progress + '%');
                progressPercentage.text(Math.round(progress) + '%');
            }, 500);
            
            console.log('Starte Restore für Backup ID:', backupId);
            
            $.ajax({
                url: jbmAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'jbm_restore_backup',
                    nonce: jbmAdmin.nonce,
                    backup_id: backupId,
                    overwrite_wpconfig: overwriteWpconfig ? 'true' : 'false',
                    keep_existing_plugins: keepExistingPlugins ? 'true' : 'false'
                },
                success: function(response) {
                    clearInterval(progressInterval);
                    progressBar.css('width', '100%');
                    progressPercentage.text('100%');
                    
                    setTimeout(function() {
                        progressContainer.fadeOut(300);
                    }, 1000);
                    
                    console.log('Restore Response:', response);
                    
                    if (response.success) {
                        // Erfolgsmeldung zusammenstellen
                        let message = response.data.message || jbmAdmin.strings.restore_successful || 'Backup erfolgreich wiederhergestellt!';
                        let migrationInfo = '';
                        
                        // URL-Migration Info
                        if (response.data.url_migrated) {
                            const fromText = jbmAdmin.strings.from_url || 'Von:';
                            const toText = jbmAdmin.strings.to_url || 'Zu:';
                            const migrationDoneText = jbmAdmin.strings.url_migration_done || '🔄 URL-Migration durchgeführt:';
                            migrationInfo = '<br><br><strong style="color: #2271b1;">' + migrationDoneText + '</strong><br>' +
                                          fromText + ' ' + response.data.old_url + '<br>' +
                                          toText + ' ' + response.data.new_url;
                        }
                        
                        // Vollbild-Erfolgsmeldung (blockiert Session-Ablauf-Dialog)
                        const successModal = $(
                            '<div style="position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.9); z-index: 999999; display: flex; align-items: center; justify-content: center;">' +
                                '<div style="background: white; padding: 40px; border-radius: 16px; max-width: 500px; text-align: center; box-shadow: 0 8px 32px rgba(0,0,0,0.3);">' +
                                    '<div style="font-size: 64px; margin-bottom: 40px;">✅</div>' +
                                    '<h2 style="margin: 0 0 15px 0; color: #00a32a; font-size: 24px;">' + (jbmAdmin.strings.restore_successful || 'Wiederherstellung erfolgreich!') + '</h2>' +
                                    '<p style="margin: 0 0 20px 0; color: #646970; font-size: 15px; line-height: 1.6;">' + message + migrationInfo + '</p>' +
                                    '<div style="padding: 15px; background: #fff3cd; border-radius: 8px; border-left: 4px solid #ffc107; margin-bottom: 20px;">' +
                                        '<p style="margin: 0; color: #856404; font-size: 14px;"><strong>' + (jbmAdmin.strings.please_relogin || 'ℹ️ Wichtig:') + '</strong><br>' + (jbmAdmin.strings.please_relogin_text || 'Bitte melden Sie sich erneut an, um fortzufahren.') + '</p>' +
                                    '</div>' +
                                    '<p style="margin: 0; color: #2271b1; font-size: 13px;">' + (jbmAdmin.strings.redirecting_login || 'Weiterleitung zur Login-Seite in %d Sekunden...').replace('%d', '<span id="bmp-countdown">120</span>') + '</p>' +
                                    '<p style="margin: 10px 0 0 0; font-size: 12px; color: #646970;">' + (jbmAdmin.strings.close_page_manual_login || 'Sie können diese Seite jetzt schließen und sich manuell neu einloggen.') + '</p>' +
                                '</div>' +
                            '</div>'
                        );
                        
                        $('body').append(successModal);
                        
                        // Countdown (2 Minuten = 120 Sekunden)
                        let countdown = 120;
                        const countdownInterval = setInterval(function() {
                            countdown--;
                            
                            // Formatiere als MM:SS
                            const minutes = Math.floor(countdown / 60);
                            const seconds = countdown % 60;
                            const displayTime = minutes + ':' + (seconds < 10 ? '0' : '') + seconds;
                            
                            $('#jbm-countdown').text(displayTime);
                            
                            if (countdown <= 0) {
                                clearInterval(countdownInterval);
                                // Zur Login-Seite weiterleiten
                                window.location.href = '/wp-login.php?redirect_to=' + encodeURIComponent(window.location.href);
                            }
                        }, 1000);
                        
                    } else {
                        clearInterval(progressInterval);
                        progressContainer.fadeOut(300);
                        
                        let errorMessage = response.data.message || 'Unbekannter Fehler beim Wiederherstellen';
                        
                        // Detaillierte Fehler-Infos anzeigen, falls vorhanden
                        if (response.data.error_details) {
                            console.error('Fehlerdetails:', response.data.error_details);
                            errorMessage += '\n\nDetails: ' + response.data.error_details.message;
                            if (response.data.error_details.file) {
                                errorMessage += '\nDatei: ' + response.data.error_details.file + ':' + response.data.error_details.line;
                            }
                        }
                        
                        // Erweiterte Fehlermeldung im Alert anzeigen
                        alert('FEHLER BEIM RESTORE:\n\n' + errorMessage + '\n\nBitte überprüfen Sie:\n' +
                              '1. Backup & Migration → Einstellungen → Aktivitätsprotokoll\n' +
                              '2. Browser-Konsole (F12) für Details\n' +
                              '3. wp-content/debug.log falls WP_DEBUG aktiv ist');
                        
                        JBM.showNotification(errorMessage, 'error');
                        button.prop('disabled', false);
                        backupItem.find('.bmp-button, .jbm-button').prop('disabled', false);
                        button.html('<span class="dashicons dashicons-backup"></span> ' + jbmAdmin.strings.restore);
                    }
                },
                error: function(xhr, status, error) {
                    clearInterval(progressInterval);
                    progressContainer.fadeOut(300);
                    
                    console.error('AJAX Error:', xhr, status, error);
                    console.error('Response Text:', xhr.responseText);
                    
                    let errorMsg = 'Kritischer Fehler beim Wiederherstellen.\n\n';
                    errorMsg += 'Status: ' + status + '\n';
                    errorMsg += 'Error: ' + error + '\n\n';
                    
                    if (xhr.responseText) {
                        try {
                            const jsonResponse = JSON.parse(xhr.responseText);
                            errorMsg += 'Details: ' + JSON.stringify(jsonResponse, null, 2);
                        } catch (e) {
                            errorMsg += 'Response (erste 500 Zeichen):\n' + xhr.responseText.substring(0, 500);
                        }
                    }
                    
                    errorMsg += '\n\nMögliche Ursachen:\n';
                    errorMsg += '- PHP Memory Limit erreicht\n';
                    errorMsg += '- PHP Execution Timeout\n';
                    errorMsg += '- Serverüberlastung\n';
                    errorMsg += '- ZIP-Archiv beschädigt\n\n';
                    errorMsg += 'Lösung: Logs überprüfen unter Einstellungen → Aktivitätsprotokoll';
                    
                    alert(errorMsg);
                    
                    JBM.showNotification('Kritischer Fehler - Siehe Browser-Konsole', 'error');
                    button.prop('disabled', false);
                    backupItem.find('.bmp-button, .jbm-button').prop('disabled', false);
                    button.html('<span class="dashicons dashicons-backup"></span> ' + jbmAdmin.strings.restore);
                }
            });
        },
        
        /**
         * Backups aktualisieren
         */
        refreshBackups: function() {
            console.log('Backups werden aktualisiert...');
            
            $.ajax({
                url: jbmAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'jbm_get_backups',
                    nonce: jbmAdmin.nonce
                },
                success: function(response) {
                    console.log('Backups-Response:', response);
                    
                    if (response.success && response.data.backups) {
                        console.log('Anzahl Backups:', response.data.backups.length);
                        JBM.renderBackups(response.data.backups);
                    } else {
                        console.log('Keine Backups gefunden oder Fehler, Seite wird neu geladen');
                        // Fallback: Seite neu laden
                        location.reload();
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Fehler beim Laden der Backups:', error);
                    location.reload();
                }
            });
        },
        
        /**
         * Backups rendern
         */
        renderBackups: function(backups) {
            const container = $('#jbm-backups-list');
            
            if (!container.length) {
                location.reload();
                return;
            }
            
            // Backup-Anzahl in der Überschrift aktualisieren
            const header = $('.jbm-backups-count-header');
            if (header.length > 0) {
                header.text(jbmAdmin.strings.existing_backups + ' (' + backups.length + ')');
            }
            
            if (backups.length === 0) {
                container.html(
                    '<div class="bmp-empty-state">' +
                    '<span class="dashicons dashicons-cloud"></span>' +
                    '<h3>' + jbmAdmin.strings.no_backups + '</h3>' +
                    '<p>' + jbmAdmin.strings.create_first_backup + '</p>' +
                    '</div>'
                );
                return;
            }
            
            let html = '';
            backups.forEach(function(backup) {
                const date = new Date(backup.backup_date);
                const formattedDate = date.toLocaleString('de-DE');
                const size = JBM.formatBytes(parseInt(backup.backup_size));
                const isProtected = backup.protected == 1 || backup.protected == '1';
                
                // Backup Item mit Schutz-Status
                html += '<div class="bmp-backup-item ' + (isProtected ? 'bmp-backup-protected' : '') + '" data-backup-id="' + backup.id + '" data-protected="' + (isProtected ? '1' : '0') + '">' +
                    '<div class="bmp-backup-icon">' +
                    '<span class="dashicons dashicons-' + (isProtected ? 'lock' : 'archive') + '" style="' + (isProtected ? 'color: #d63638;' : '') + '"></span>' +
                    '</div>' +
                    '<div class="bmp-backup-info">' +
                    '<h4>' + JBM.escapeHtml(backup.backup_name);
                
                // Protected Badge
                if (isProtected) {
                    html += '<span class="jbm-protected-badge" style="display: inline-block; margin-left: 8px; padding: 2px 8px; background: #d63638; color: white; border-radius: 3px; font-size: 11px; font-weight: 600;">🔒 ' + jbmAdmin.strings.protected + '</span>';
                }
                
                html += '</h4>' +
                    '<div class="bmp-backup-meta">' +
                    '<span class="bmp-backup-type">' +
                    '<span class="dashicons dashicons-tag"></span>' +
                    JBM.escapeHtml(backup.backup_type) +
                    '</span>' +
                    '<span class="bmp-backup-size">' +
                    '<span class="dashicons dashicons-chart-bar"></span>' +
                    size +
                    '</span>' +
                    '<span class="bmp-backup-date">' +
                    '<span class="dashicons dashicons-calendar"></span>' +
                    formattedDate +
                    '</span>' +
                    '</div>' +
                    '</div>' +
                    '<div class="bmp-backup-actions">' +
                    '<button class="bmp-button bmp-button-small ' + (isProtected ? 'bmp-button-warning' : 'bmp-button-secondary') + ' jbm-toggle-protection" data-backup-id="' + backup.id + '" title="' + (isProtected ? jbmAdmin.strings.remove_protection : jbmAdmin.strings.protect_from_deletion) + '">' +
                    '<span class="dashicons dashicons-' + (isProtected ? 'unlock' : 'lock') + '"></span>' +
                    '<span class="protection-text">' + (isProtected ? jbmAdmin.strings.unlock : jbmAdmin.strings.protect) + '</span>' +
                    '</button>' +
                    '<button class="bmp-button bmp-button-small bmp-button-success jbm-restore-backup" data-backup-id="' + backup.id + '">' +
                    '<span class="dashicons dashicons-backup"></span>' +
                    jbmAdmin.strings.restore +
                    '</button>' +
                    '<a href="' + jbmAdmin.ajaxUrl + '?action=jbm_download_backup&backup_id=' + backup.id + '&nonce=' + jbmAdmin.nonce + '" class="bmp-button bmp-button-small bmp-button-secondary jbm-download-backup" data-backup-id="' + backup.id + '" data-backup-size="' + backup.backup_size + '">' +
                    '<span class="dashicons dashicons-download"></span>' +
                    jbmAdmin.strings.download +
                    '</a>' +
                    '<button class="bmp-button bmp-button-small bmp-button-danger jbm-delete-backup" data-backup-id="' + backup.id + '" ' + (isProtected ? 'disabled title="' + jbmAdmin.strings.protected_cannot_delete + '"' : '') + '>' +
                    '<span class="dashicons dashicons-trash"></span>' +
                    jbmAdmin.strings.delete +
                    '</button>' +
                    '</div>' +
                    '</div>';
            });
            
            container.html(html);
        },
        
        /**
         * Bytes formatieren
         */
        formatBytes: function(bytes, precision) {
            precision = precision || 2;
            const units = ['B', 'KB', 'MB', 'GB', 'TB'];
            let i = 0;
            
            while (bytes >= 1024 && i < units.length - 1) {
                bytes /= 1024;
                i++;
            }
            
            return bytes.toFixed(precision) + ' ' + units[i];
        },
        
        /**
         * HTML escapen
         */
        escapeHtml: function(text) {
            const map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return text.replace(/[&<>"']/g, function(m) { return map[m]; });
        },
        
        /**
         * Holt maximale Upload-Größe (Fallback)
         */
        getMaxUploadSize: function() {
            return '512 MB'; // Standardwert, sollte vom Server kommen
        },
        
        /**
         * Zeitplan speichern
         */
        saveSchedule: function(e) {
            e.preventDefault();
            
            const form = $(e.target);
            const button = form.find('button[type="submit"]');
            
            button.prop('disabled', true);
            
            $.ajax({
                url: jbmAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'jbm_save_schedule',
                    nonce: jbmAdmin.nonce,
                    enabled: $('#schedule-enabled').is(':checked'),
                    frequency: $('#schedule-frequency').val(),
                    time: $('#schedule-time').val(),
                    backup_type: $('#schedule-backup-type').val()
                },
                success: function(response) {
                    if (response.success) {
                        JBM.showNotification(response.data.message, 'success');
                        setTimeout(function() {
                            location.reload();
                        }, 1500);
                    } else {
                        JBM.showNotification(response.data.message, 'error');
                        button.prop('disabled', false);
                    }
                },
                error: function() {
                    JBM.showNotification('Fehler beim Speichern', 'error');
                    button.prop('disabled', false);
                }
            });
        },
        
        /**
         * Migration starten
         */
        startMigration: function(e) {
            e.preventDefault();
            
            const form = $(e.target);
            const button = form.find('button[type="submit"]');
            const progressContainer = $('#jbm-migration-progress');
            
            button.prop('disabled', true);
            form.fadeOut(300, function() {
                progressContainer.fadeIn(300);
            });
            
            // Schritte animieren
            let currentStep = 0;
            const steps = progressContainer.find('.jbm-progress-step');
            const stepInterval = setInterval(function() {
                if (currentStep < steps.length) {
                    steps.eq(currentStep).addClass('active');
                    currentStep++;
                }
            }, 2000);
            
            $.ajax({
                url: jbmAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'jbm_migrate_site',
                    nonce: jbmAdmin.nonce,
                    old_url: $('#old-url').val(),
                    new_url: $('#new-url').val()
                },
                success: function(response) {
                    clearInterval(stepInterval);
                    steps.addClass('active');
                    
                    setTimeout(function() {
                        if (response.success) {
                            JBM.showNotification(response.data.message, 'success');
                            setTimeout(function() {
                                window.location.href = $('#new-url').val() + '/wp-admin/';
                            }, 2000);
                        } else {
                            JBM.showNotification(response.data.message, 'error');
                            progressContainer.fadeOut(300, function() {
                                form.fadeIn(300);
                                button.prop('disabled', false);
                                steps.removeClass('active');
                            });
                        }
                    }, 1000);
                },
                error: function() {
                    clearInterval(stepInterval);
                    JBM.showNotification('Fehler bei der Migration', 'error');
                    progressContainer.fadeOut(300, function() {
                        form.fadeIn(300);
                        button.prop('disabled', false);
                        steps.removeClass('active');
                    });
                }
            });
        },
        
        /**
         * Kompressionswert aktualisieren
         */
        updateCompressionValue: function() {
            const value = $('#compression-level').val();
            $('#compression-value').text(value);
        },
        
        /**
         * Logs löschen
         */
        clearLogs: function() {
            const modal = $('#jbm-clear-logs-modal');
            const title = $('#jbm-clear-logs-modal-title');
            const message = $('#jbm-clear-logs-modal-message');
            const cancelBtn = $('#jbm-clear-logs-modal-cancel .cancel-text');
            const confirmBtn = $('#jbm-clear-logs-modal-confirm .confirm-text');
            
            // Modal mit übersetzten Texten füllen
            title.text(jbmAdmin.strings.clear_logs_title || 'Logs wirklich löschen?');
            message.text(jbmAdmin.strings.clear_logs_message || 'Möchten Sie wirklich alle Logs löschen?');
            cancelBtn.text(jbmAdmin.strings.clear_logs_cancel_button || 'Abbrechen');
            confirmBtn.text(jbmAdmin.strings.clear_logs_confirm_button || 'Ja, löschen');
            
            // Modal anzeigen
            modal.css('display', 'flex');
            
            // Cancel-Button Handler
            $('#jbm-clear-logs-modal-cancel').off('click').on('click', function() {
                modal.hide();
            });
            
            // Confirm-Button Handler
            $('#jbm-clear-logs-modal-confirm').off('click').on('click', function() {
                const button = $(this);
                button.prop('disabled', true);
                
                $.ajax({
                    url: jbmAdmin.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'jbm_clear_logs',
                        nonce: jbmAdmin.nonce
                    },
                    success: function(response) {
                        modal.hide();
                        if (response.success) {
                            JBM.showNotification(jbmAdmin.strings.logs_cleared || 'Logs wurden erfolgreich gelöscht', 'success');
                            setTimeout(function() {
                                location.reload();
                            }, 1000);
                        } else {
                            JBM.showNotification(response.data.message || jbmAdmin.strings.error_clearing_logs || 'Fehler beim Löschen der Logs', 'error');
                            button.prop('disabled', false);
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('AJAX Error:', xhr, status, error);
                        modal.hide();
                        JBM.showNotification(jbmAdmin.strings.error_clearing_logs || 'Fehler beim Löschen der Logs', 'error');
                        button.prop('disabled', false);
                    }
                });
            });
        },
        
        /**
         * Benachrichtigung anzeigen
         */
        showNotification: function(message, type) {
            type = type || 'info';
            
            const icons = {
                success: 'yes',
                error: 'warning',
                warning: 'warning',
                info: 'info'
            };
            
            const notification = $('<div>')
                .addClass('bmp-alert bmp-alert-' + type)
                .html(
                    '<span class="dashicons dashicons-' + icons[type] + '"></span>' +
                    '<div>' + message + '</div>'
                )
                .css({
                    position: 'fixed',
                    top: '32px',
                    right: '20px',
                    zIndex: 999999,
                    minWidth: '300px',
                    maxWidth: '500px',
                    boxShadow: '0 4px 16px rgba(0,0,0,0.2)',
                    animation: 'slideInRight 0.3s ease'
                });
            
            $('body').append(notification);
            
            setTimeout(function() {
                notification.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 5000);
        },
        
        /**
         * Zeigt motivierende Nachrichten & Backup-Tipps während der Erstellung
         */
        showBackupMessage: function(index) {
            const messages = [
                {
                    icon: '🚀',
                    text: jbmAdmin.strings.backup_creating || 'Dein Backup wird gerade erstellt...',
                    tip: jbmAdmin.strings.backup_tip_regular || 'Regelmäßige Backups sind der beste Schutz gegen Datenverlust!'
                },
                {
                    icon: '⏳',
                    text: jbmAdmin.strings.backup_taking_time || 'Es dauert noch ein bisschen...',
                    tip: jbmAdmin.strings.backup_tip_hacked || 'Wusstest du? 60% aller WordPress-Websites wurden bereits gehackt oder hatten Datenverlust.'
                },
                {
                    icon: '💪',
                    text: jbmAdmin.strings.backup_patience || 'Bitte noch etwas Geduld...',
                    tip: jbmAdmin.strings.backup_tip_save_work || 'Ein gutes Backup kann Stunden oder Tage Arbeit retten!'
                },
                {
                    icon: '⚡',
                    text: jbmAdmin.strings.backup_almost_done || 'Noch ein wenig, gleich ist es geschafft...',
                    tip: jbmAdmin.strings.backup_tip_321 || 'Wusstest du? Die 3-2-1 Regel: 3 Kopien, 2 Medien, 1 extern.'
                },
                {
                    icon: '🔒',
                    text: jbmAdmin.strings.backup_fast_finished || 'Fast fertig...',
                    tip: jbmAdmin.strings.backup_tip_test || 'Tipp: Teste deine Backups regelmäßig durch Wiederherstellung auf einer Test-Domain!'
                },
                {
                    icon: '📦',
                    text: jbmAdmin.strings.backup_last_step || 'Letzter Schritt...',
                    tip: jbmAdmin.strings.backup_tip_bankrupt || 'Wusstest du? 93% der Unternehmen ohne Backup gehen nach einem Datenverlust bankrott.'
                },
                {
                    icon: '🛡️',
                    text: jbmAdmin.strings.backup_saving_data || 'Deine Daten werden gesichert...',
                    tip: jbmAdmin.strings.backup_tip_updates || 'WordPress-Updates können manchmal schiefgehen - ein Backup ist Gold wert!'
                },
                {
                    icon: '💾',
                    text: jbmAdmin.strings.backup_creating_archive || 'Backup-Archiv wird erstellt...',
                    tip: jbmAdmin.strings.backup_tip_external || 'Tipp: Lade wichtige Backups herunter und bewahre sie extern auf (Google Drive, Dropbox).'
                },
                {
                    icon: '🌟',
                    text: jbmAdmin.strings.backup_running || 'Sicherung läuft...',
                    tip: jbmAdmin.strings.backup_tip_plugins || 'Wusstest du? Plugin-Updates sind die häufigste Ursache für Website-Crashes.'
                },
                {
                    icon: '🎯',
                    text: jbmAdmin.strings.backup_almost_there || 'Gleich geschafft...',
                    tip: jbmAdmin.strings.backup_tip_pro_tip || 'Profi-Tipp: Erstelle vor jedem großen Update ein vollständiges Backup!'
                }
            ];
            
            // Bei Erfolg spezielle Nachricht
            if (index === 'success') {
                $('#jbm-backup-messages').css('background', 'linear-gradient(135deg, #34C759 0%, #28A745 100%)');
                $('#jbm-message-icon').text('✅').css('animation', 'none');
                $('#jbm-message-text').text(jbmAdmin.strings.backup_success || 'Backup erfolgreich erstellt!');
                $('#jbm-message-tip').text(jbmAdmin.strings.backup_success_tip || 'Deine Daten sind jetzt sicher gesichert. 🎉');
                return;
            }
            
            // Bei Fehler
            if (index === 'error') {
                $('#jbm-backup-messages').css('background', 'linear-gradient(135deg, #ff6b6b 0%, #c92a2a 100%)');
                $('#jbm-message-icon').text('❌').css('animation', 'none');
                $('#jbm-message-text').text(jbmAdmin.strings.backup_error || 'Ein Fehler ist aufgetreten!');
                $('#jbm-message-tip').text(jbmAdmin.strings.backup_error_tip || 'Bitte prüfen Sie die Logs oder kontaktieren Sie den Support.');
                return;
            }
            
            // Normalen Gradient wiederherstellen
            $('#jbm-backup-messages').css('background', 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)');
            $('#jbm-message-icon').css('animation', 'pulse 2s ease-in-out infinite');
            
            // Normale Nachrichten rotieren
            const message = messages[index % messages.length];
            
            // Fade-Animation
            $('#jbm-message-content').fadeOut(200, function() {
                $('#jbm-message-icon').text(message.icon);
                $('#jbm-message-text').text(message.text);
                $('#jbm-message-tip').text(message.tip);
                $(this).fadeIn(200);
            });
        },
        
        /**
         * Zeigt Rate-Limit-Modal zentriert an
         * Shows rate limit modal centered
         */
        showRateLimitModal: function(data) {
            // Entferne eventuell vorhandenes Modal
            $('#jbm-rate-limit-modal').remove();
            
            const modalHtml = '<div id="jbm-rate-limit-modal" style="display: flex; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.6); z-index: 999999; justify-content: center; align-items: center;">' +
                '<div style="background: white; padding: 40px; border-radius: 12px; max-width: 500px; width: 90%; box-shadow: 0 10px 40px rgba(0,0,0,0.3); text-align: center;">' +
                '<div style="font-size: 56px; margin-bottom: 40px;">⏳</div>' +
                '<h3 style="margin: 0 0 15px 0; font-size: 22px; color: #1d2327;">' + data.message + '</h3>' +
                '<p style="margin: 0 0 25px 0; color: #666; line-height: 1.6; font-size: 14px;">' +
                '<span class="dashicons dashicons-shield" style="color: #2271b1; margin-right: 5px;"></span>' +
                data.reason + '</p>' +
                '<div style="background: #f0f6fc; padding: 15px; border-radius: 8px; margin-bottom: 20px;">' +
                '<span style="font-size: 32px; font-weight: 700; color: #2271b1;" id="jbm-countdown">' + data.wait_time + '</span>' +
                '<span style="color: #666; margin-left: 5px;">' + (jbmAdmin.strings.seconds_remaining || 'Sekunden verbleibend') + '</span>' +
                '</div>' +
                '<button type="button" id="jbm-rate-limit-close" class="button button-primary" style="padding: 10px 30px;">' +
                (jbmAdmin.strings.understood || 'Verstanden') + '</button>' +
                '</div></div>';
            
            $('body').append(modalHtml);
            
            // Countdown - schließt automatisch wenn Zeit abgelaufen
            let countdown = data.wait_time;
            const countdownInterval = setInterval(function() {
                countdown--;
                if (countdown <= 0) {
                    clearInterval(countdownInterval);
                    $('#jbm-countdown').text('0');
                    // Automatisch schließen wenn Zeit abgelaufen
                    setTimeout(function() {
                        $('#jbm-rate-limit-modal').fadeOut(300, function() {
                            $(this).remove();
                        });
                    }, 500);
                } else {
                    $('#jbm-countdown').text(countdown);
                }
            }, 1000);
            
            // Schließen bei Klick auf Button
            $('#jbm-rate-limit-close').on('click', function() {
                $('#jbm-rate-limit-modal').fadeOut(300, function() {
                    $(this).remove();
                    clearInterval(countdownInterval);
                });
            });
            
            // Schließen bei Klick außerhalb
            $('#jbm-rate-limit-modal').on('click', function(e) {
                if (e.target === this) {
                    $(this).fadeOut(300, function() {
                        $(this).remove();
                        clearInterval(countdownInterval);
                    });
                }
            });
        },
        
        /**
         * Aktualisiert den Backup-Status in der Admin-Bar (Live-Update)
         * Updates the backup status in admin bar (live update)
         */
        updateAdminBarStatus: function() {
            $.ajax({
                url: jbmAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'jbm_get_backup_status',
                    nonce: jbmAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        const statusNode = $('#wp-admin-bar-jbm-backup-status');
                        if (statusNode.length) {
                            // Format: Jenva Backup | Status (gesamter Text farbig, nur Status animiert)
                            let fullHtml;
                            
                            if (response.data.animated) {
                                fullHtml = '<span style="color: ' + response.data.color + '; font-weight: 600;">Jenva Backup <span style="margin: 0 6px;">|</span> <span class="jbm-status-animated">' + response.data.status_text + '</span></span>';
                            } else {
                                fullHtml = '<span style="color: ' + response.data.color + '; font-weight: 600;">Jenva Backup <span style="margin: 0 6px;">|</span> ' + response.data.status_text + '</span>';
                            }
                            
                            const link = statusNode.find('.ab-item');
                            link.html(fullHtml);
                            
                            // Aktualisiere die CSS-Klasse
                            statusNode.removeClass('jbm-backup-status-ok jbm-backup-status-warning')
                                      .addClass(response.data.class);
                            
                            // Aktualisiere den Tooltip
                            link.attr('title', response.data.message);
                            
                            console.log('Admin-Bar Status aktualisiert:', response.data.status_text);
                        }
                    }
                }
            });
        }
    };
    
    // Initialisieren, wenn DOM bereit ist
    $(document).ready(function() {
        console.log('DOM ready - Initialisiere JBM...');
        console.log('jQuery verfügbar:', typeof $ !== 'undefined');
        console.log('jbmAdmin verfügbar:', typeof jbmAdmin !== 'undefined');
        if (typeof jbmAdmin !== 'undefined') {
            console.log('jbmAdmin.ajaxUrl:', jbmAdmin.ajaxUrl);
            console.log('jbmAdmin.nonce:', jbmAdmin.nonce);
        }
        try {
            JBM.init();
        } catch (error) {
            console.error('FEHLER bei JBM.init():', error);
            alert('JavaScript-Fehler: ' + error.message + '\n\nBitte Browser-Konsole (F12) für Details prüfen.');
        }
    });
    
    // Animation für Benachrichtigungen
    $('<style>')
        .prop('type', 'text/css')
        .html(`
            @keyframes slideInRight {
                from {
                    transform: translateX(400px);
                    opacity: 0;
                }
                to {
                    transform: translateX(0);
                    opacity: 1;
                }
            }
        `)
        .appendTo('head');
    
})(jQuery);

